﻿using NAudio.Dsp;
using NAudio.Wave;
using Spectral1.DATA_ACCESS;
using System;
using System.Collections.Generic;
using System.Drawing;

namespace Spectral1.SPECTRUM_ANALYSER
{
    class c_spectrum_analyser : WaveStream
    {
        //Excellent : https://www.dspguide.com/ch12/2.htm
        //https://markheath.net/post/looped-playback-in-net-with-naudio
        //https://download.ni.com/evaluation/pxi/Understanding%20FFTs%20and%20Windowing.pdf

        #region "======================= DECLARATIONS ========================="
        public const int fftLength = 8192;
        public const int fft_running_av_size = 10;
        public const int fft_sample_rate = 44100;
        public const int horiz_axis_y_offset = 40;

        private WaveStream sourceStream;
        private int m = (int)Math.Log(fftLength, 2.0);
        private Complex[] _fftBuffer = new Complex[fftLength];
        private int[] fft_running_av = new int[fftLength];
        private Point[] _waveform_signal_display;
        private double spectrum_horiz_pixels_per_fft_step;
        private double fft_steps_per_horiz_pixel;
        private double waveform_signal_horiz_position = 0;
        private int last_int_waveform_signal_horiz_position = 0;

        private double _spectrum_display_horiz_pixels = 0;
        private double _spectrum_display_vertical_pixels = 0;
        private int _spectrum_display_vertical_centre_y;

        private double _signal_display_horiz_pixels = 0;
        private double _signal_display_vertical_pixels = 0;
        private int _signal_display_vertical_centre_y;

        private double _analysis_display_horiz_pixels = 0;
        private double _analysis_display_vertical_pixels = 0;
        private int _analysis_display_vertical_centre_y;

        private int fftPos = 0;
        public event EventHandler<FftEventArgs> FftCalculated;
        private FftEventArgs fftArgs =  new FftEventArgs();
        private long _loop_start_position;
        private long _loop_end_position;
        private double _loop_duration_s;
        private loop_statuses loop_status = loop_statuses.not_looping;
        private bool last_signal_polarity = true;
        private List<int> timbre_harmonics;
        private enum loop_statuses
        {
            not_looping,
            seeking_start_zero_cross_point,
            seeking_end_zero_cross_point,
            looping
        }

        #endregion

        #region "======================= PROPERTIES ========================="
        public override WaveFormat WaveFormat
        {
            get { return sourceStream.WaveFormat; }
        }

        public override long Length
        {
            get { return sourceStream.Length; }
        }

        public override long Position
        {
            get { return sourceStream.Position; }
            set { sourceStream.Position = value; }
        }

       public Complex[] fftBuffer
       {
            get { return _fftBuffer; }
       }

        public Point[] waveform_signal_display_buffer
        {
            get { return _waveform_signal_display; }
        }

        public List<int> result_timbre_harmonics
        {
            get { return timbre_harmonics; }
        }

        #endregion

        #region "======================= METHODS ========================="
        public c_spectrum_analyser(WaveStream sourceStream,Rectangle spectrum_display_rectangle, Rectangle signal_display_rectangle, Rectangle analysis_display_rectangle)
        {
            this.sourceStream = sourceStream;
            _spectrum_display_horiz_pixels = spectrum_display_rectangle.Width;
            _spectrum_display_vertical_pixels = spectrum_display_rectangle.Height;
            _spectrum_display_vertical_centre_y = spectrum_display_rectangle.Height/2;

            _signal_display_horiz_pixels = signal_display_rectangle.Width;
            _signal_display_vertical_pixels = signal_display_rectangle.Height;
            _signal_display_vertical_centre_y = signal_display_rectangle.Height / 2;

            _analysis_display_horiz_pixels = analysis_display_rectangle.Width;
            _analysis_display_vertical_pixels = analysis_display_rectangle.Height;
            _analysis_display_vertical_centre_y = analysis_display_rectangle.Height / 2;

            spectrum_horiz_pixels_per_fft_step =  (double)_spectrum_display_horiz_pixels / (double)fftLength;
            _waveform_signal_display = new Point[signal_display_rectangle.Width];

            timbre_harmonics = new List<int>();
            for (int i = 0; i < DA_Spectral.max_harmonics; i++) { timbre_harmonics.Add(0); }
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            int read_bytes = sourceStream.Read(buffer, offset, count);

            //read contains how many bytes are read from the stream and there are 4 bytes per sample for 32 bit.
            for (int sample_id = 0; sample_id < read_bytes / 4; sample_id++)
                {
                    //Get sample from buffer
                    float sample = BitConverter.ToSingle(buffer, sample_id * 4);

                    //add to FFT
                    add_fft_sample(sample);

                    bool this_signal_polarity = (sample > 0);
                    if (this_signal_polarity != last_signal_polarity)
                    {
                        switch (loop_status)
                        {
                            case loop_statuses.seeking_start_zero_cross_point:
                                _loop_start_position = sourceStream.Position;
                                _loop_end_position = (long)(_loop_start_position + (352800.0 * _loop_duration_s));
                                loop_status = loop_statuses.seeking_end_zero_cross_point;
                                break;
                            case loop_statuses.seeking_end_zero_cross_point:
                                if(sourceStream.Position > _loop_end_position)
                                _loop_end_position = sourceStream.Position;
                            loop_status = loop_statuses.looping;
                            break;
                        }
                    }
                    last_signal_polarity = this_signal_polarity;
                }

            if ((loop_status == loop_statuses.looping) && (sourceStream.Position > _loop_end_position))
            {
                sourceStream.Position = _loop_start_position;
            }

            return read_bytes;
        }

        private void add_fft_sample(float value)
        {
            if (FftCalculated == null) return;

            //Add to FFT
            _fftBuffer[fftPos].X = (float)(value * FastFourierTransform.HammingWindow(fftPos, fftLength));
            _fftBuffer[fftPos].Y = 0;
            fftPos++;

            //Add to signal display
            const double waveform_signal_scaling = 75;
            int idp = (int)(waveform_signal_horiz_position);
            if (idp != last_int_waveform_signal_horiz_position)
            {
                _waveform_signal_display[idp] = new Point((int)(waveform_signal_horiz_position), _signal_display_vertical_centre_y + ((int)(value* waveform_signal_scaling)));
                last_int_waveform_signal_horiz_position = idp;
            }
            waveform_signal_horiz_position += spectrum_horiz_pixels_per_fft_step;

            if (fftPos >= fftLength)
            {
                fftPos = 0;
                waveform_signal_horiz_position = 0;
                FastFourierTransform.FFT(true, m, _fftBuffer);
                FftCalculated(this, fftArgs);
            }
        }

        public int db(double signal)
        {
            if (signal < 1)
            { return 0; }
            else
            { return (int)(100 * Math.Log10(signal)); }
        }

        public Point[] get_spectrum_polyline(bool display_low_freq_only)
        {
            if (display_low_freq_only)
            { fft_steps_per_horiz_pixel = (double)fftLength / ((double)_spectrum_display_horiz_pixels * 16.0); }
            else
            { fft_steps_per_horiz_pixel = (double)fftLength / ((double)_spectrum_display_horiz_pixels * 8.0); }
            
            Point[] result = new Point[(int)_spectrum_display_horiz_pixels + 2];
            int y_base = (int)(_spectrum_display_vertical_pixels - horiz_axis_y_offset);
            for (int i = 0; i < _spectrum_display_horiz_pixels; i++)
            {
                int fft_buffer_index = (int)(i * fft_steps_per_horiz_pixel);
                int y = Math.Abs((int)(_fftBuffer[fft_buffer_index].X * 2700));

                fft_running_av[i] = (((fft_running_av_size - 1) * fft_running_av[i]) + y)/ fft_running_av_size;
                result[i] = new Point(i, y_base - db(y));
            }
            result[(int)_spectrum_display_horiz_pixels] = new Point((int)_spectrum_display_horiz_pixels, y_base);
            result[(int)_spectrum_display_horiz_pixels + 1] = new Point(0, y_base);
            return result;
        }

        public Point[] get_spectrum_ra_polyline(bool display_low_freq_only)
        {
            Point[] result = new Point[(int)_spectrum_display_horiz_pixels + 2];
            int y_base = (int)(_spectrum_display_vertical_pixels - horiz_axis_y_offset);
            for (int i = 0; i < _spectrum_display_horiz_pixels; i++)
            {
                result[i] = new Point(i, y_base - db(fft_running_av[i]));
            }
            result[(int)_spectrum_display_horiz_pixels] = new Point((int)_spectrum_display_horiz_pixels, y_base);
            result[(int)_spectrum_display_horiz_pixels + 1] = new Point(0, y_base);
            return result;
        }

        private int get_max_of_merged_bins(int centre_bin_index,int neighbour_count)
        {
            if (centre_bin_index < neighbour_count) return fft_running_av[centre_bin_index];
            if (centre_bin_index >= fftLength - neighbour_count) return fft_running_av[centre_bin_index];

            int max_level = 0;
            for (int i = centre_bin_index - neighbour_count; i <= centre_bin_index + neighbour_count; i++)
            {
                if (fft_running_av[i] > max_level)
                { max_level = fft_running_av[i]; }
            }

            return max_level;
        }

        public Point[] get_spectrum_analysis_polyline()
        {
            const int column_width = 8;
            const int plot_margin = 20;

            //Find peak fequency
            double max_level = 0;
            int f_index_of_fundamental = 0;
            double x_step = (_analysis_display_horiz_pixels - 2 * plot_margin) / (double)DA_Spectral.max_harmonics;
            for (int i = 0; i < fftLength;i++)
            {
                if (fft_running_av[i] > max_level)
                {
                    max_level = fft_running_av[i];
                    f_index_of_fundamental = i;
                }
            }

            int neighbour_count = f_index_of_fundamental / 5;
            if (neighbour_count > 20) { neighbour_count = 20; }

            //Adjust fundamental if there is a preceding harmonic 
            if (get_max_of_merged_bins((int)(f_index_of_fundamental / 2), neighbour_count) > (max_level/10.0))
            {
                f_index_of_fundamental = (int)(f_index_of_fundamental / 2);
            }
            
            Point[] result = new Point[(DA_Spectral.max_harmonics * 4) + 1];
            int y_base = (int)(_analysis_display_vertical_pixels - horiz_axis_y_offset);
            int p_index = 0;
            int f_index;
            int x;
            int y;
            for (int i = 0; i < DA_Spectral.max_harmonics; i++)
            {
                f_index = f_index_of_fundamental * (i + 1);

                x = (int)(x_step * i) + plot_margin;

                if (f_index <= fftLength)
                {
                    y = db(get_max_of_merged_bins(f_index, neighbour_count));
                }
                else
                {
                    y = 0;
                }
                timbre_harmonics[i] = y;

                result[p_index] = new Point(x, y_base);
                p_index++;
                result[p_index] = new Point(x, y_base - y);
                p_index++;
                result[p_index] = new Point(x + column_width, y_base - y);
                p_index++;
                result[p_index] = new Point(x + column_width, y_base);
                p_index++;

            }

            result[p_index] = new Point(0, y_base);

            return result;
        }

        public void start_loop(double loop_duration_s)
        {
            loop_status = loop_statuses.seeking_start_zero_cross_point;
            _loop_duration_s = loop_duration_s;
        }

        public void stop_loop()
        {
            loop_status = loop_statuses.not_looping;
        }

        #endregion

    }

    public class FftEventArgs : EventArgs
    {
        public FftEventArgs()
        {}  
    }
}
